unit MirrPanl;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes,
  Graphics, Controls, Forms, Menus, ExtCtrls;

type
  TMirrorPanel = class(TCustomPanel)
  private
    FMirrorX: Boolean;
    FMirrorY: Boolean;
    FPicture: TBitmap;
    FStretch: Boolean;
    NoPaint: Boolean;
    Painting: Boolean;
    procedure PaintNow;
    procedure SetMirrorX(Value: Boolean);
    procedure SetMirrorY(Value: Boolean);
    procedure SetPicture(Value: TBitmap);
    procedure SetStretch(Value: Boolean);
  protected
    procedure Paint; override;
    procedure WndProc(var Message: TMessage); override;
  public
    constructor Create(AOwner:TComponent); override;
    destructor Destroy; override;
  published
    property MirrorX: Boolean read FMirrorX write SetMirrorX;
    property MirrorY: Boolean read FMirrorY write SetMirrorY;
    property Align;
    property Alignment;
    property BevelInner;
    property BevelOuter;
    property BevelWidth;
    property BorderWidth;
    property BorderStyle;
    property DragCursor;
    property DragMode;
    property Enabled;
    property Caption;
    property Color;
    property Ctl3D;
    property Font;
    property Locked;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property Picture: TBitMap read FPicture write SetPicture;
    property PopupMenu;
    property ShowHint;
    property Stretch: Boolean read FStretch write SetStretch;
    property TabOrder;
    property TabStop;
    property Visible;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnResize;
end;

procedure Register;

implementation

constructor TMirrorPanel.Create(AOwner:TComponent);
begin
inherited Create(AOwner);
FPicture := TBitmap.Create;
end;

destructor TMirrorPanel.Destroy;
begin
FPicture.Free;
inherited Destroy;
end;

procedure TMirrorPanel.WndProc(var Message: TMessage);
begin
if Message.Msg = WM_PAINT then
  PaintNow;
inherited WndProc(Message);
end;

procedure TMirrorPanel.PaintNow;
var
  i: Integer;
  OldPal: HPalette;
  Bmp: TBitmap;
  Rect: TRect;
  Header, Bits: Pointer;
  HeaderSize: Integer;
  BitsSize : LongInt;
  TopColor, BottomColor: TColor;
  Text: array[0..255] of Char;
  FontHeight: Integer;
const
  Alignments: array[TAlignment] of Word = (DT_LEFT, DT_RIGHT, DT_CENTER);

procedure MemFree(Ptr: Pointer; Size: Longint);
begin
  if Size < $FFFF then
    FreeMem(Ptr, Size)
  else
    GlobalFreePtr(Ptr);
end;

procedure AdjustColors(Bevel: TPanelBevel);
begin
TopColor := clBtnHighlight;
if Bevel = bvLowered then
  TopColor := clBtnShadow;
BottomColor := clBtnShadow;
if Bevel = bvLowered then
  BottomColor := clBtnHighlight;
end;

begin
if Nopaint or Painting then
  Exit;
NoPaint := True;
Painting := True;
Bmp := TBitmap.Create;
Bmp.Height := Height;
Bmp.Width := Width;
Rect := GetClientRect;
with Bmp.Canvas do
  begin
  Brush.Color := Self.Color;
  Brush.Style := bsSolid;
  if (FStretch = True) and (not FPicture.Empty) then
    StretchDraw(ClientRect, FPicture)
  else
    begin
    FillRect(Rect);
    if not (FPicture.Empty) then
      Draw(0, 0, FPicture);
    end;
  Brush.Style := bsClear;
  Font := Self.Font;
  FontHeight := TextHeight('W');
  with Rect do
    begin
    Top := ((Bottom + Top) - FontHeight) shr 1;
    Bottom := Top + FontHeight;
    end;
  StrPCopy(Text, Caption);
  DrawText(Handle, Text, StrLen(Text), Rect, (DT_EXPANDTABS or
    DT_VCENTER) or Alignments[Alignment]);
  Rect := GetClientRect;
  try
  	GetDIBSizes(Bmp.Handle, HeaderSize, BitsSize);
  	Header := MemAlloc(HeaderSize);
  	Bits := MemAlloc(BitsSize);
    GetDIB(Bmp.Handle, FPicture.Palette, Header^, Bits^);
    SetStretchBltMode(Handle, STRETCH_DELETESCANS);
    if (FMirrorX and FMirrorY) then
      StretchDIBits(Handle, 0, 0, Width, Height,
      Width, Height, -Width, -Height, Bits, TBitmapInfo(Header^),
      DIB_RGB_COLORS, SRCCOPY)
  	else if FMirrorX then
    	StretchDIBits(Handle, 0, 0, Width, Height,
        Width, 0, -Width, Height, Bits, TBitmapInfo(Header^),
        DIB_RGB_COLORS, SRCCOPY)
  	else if FMirrorY then
      StretchDIBits(Handle, 0, 0, Width, Height,
        0, Height, Width, -Height, Bits, TBitmapInfo(Header^),
        DIB_RGB_COLORS, SRCCOPY);
  finally
    MemFree(Header, HeaderSize);
    MemFree(Bits, BitsSize);
    end;
  end;
with Canvas do
  begin
  CopyRect(ClientRect, Bmp.Canvas, ClientRect);
  for i := 0 to Self.ControlCount - 1 do
     Self.Controls[i].Refresh;
  if BevelOuter <> bvNone then
    begin
    AdjustColors(BevelOuter);
    Frame3D(Canvas, Rect, TopColor, BottomColor, BevelWidth);
    end;
  Frame3D(Canvas, Rect, Color, Color, BorderWidth);
  if BevelInner <> bvNone then
    begin
    AdjustColors(BevelInner);
    Frame3D(Canvas, Rect, TopColor, BottomColor, BevelWidth);
    end;
  end;
Bmp.Free;
Painting := False;
end;

procedure TMirrorPanel.Paint;
begin
NoPaint := False;
PaintNow;
end;

procedure TMirrorPanel.SetMirrorX(Value: Boolean);
begin
if Value <> FMirrorX then
  begin
  FMirrorX := Value;
  NoPaint := False;
  Invalidate;
  end;
end;

procedure TMirrorPanel.SetMirrorY(Value: Boolean);
begin
if Value <> FMirrorY then
  begin
  FMirrorY := Value;
  NoPaint := False;
  Invalidate;
  end;
end;

procedure TMirrorPanel.SetPicture(Value: TBitmap);
begin
FPicture.Assign(Value);
NoPaint := False;
Invalidate;
end;

procedure TMirrorPanel.SetStretch(Value: Boolean);
begin
if Value <> FStretch then
  begin
  FStretch := Value;
  NoPaint := False;
  Invalidate;
  end;
end;

procedure Register;
begin
RegisterComponents('Samples',[TMirrorPanel]);
end;

end.

 